/****************************************************************************
 *   $Id:: main.c 6954 2011-03-23 23:02:28Z usb00423                      $
 *   Project: NXP LPC11U6x IAP
 *
 *   Description:
 *     This file contains the In-Application Programming code for LPC 11U6x.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
/* LPC11U6xx definitions */
#include "LPC11U6x.h"
#include "lpc11U6x_uart.h"
#include "lpc11U6x_clkconfig.h"
#include "IAP.h"
#include "EEPROM.h"
#include <stdio.h>
#include <string.h>

#define TEST_TIMER_NUM		0		/* 0 or 1 for 16-bit timers only */
#define EEPROM_SIZE   		4096 - 64
#define SRC0 0x1FFF2100 // source address if RAM

#define SRC1 0x1FFF2100 // source address if RAM
#define DST1 0x012100		// destination address of flash

extern volatile uint32_t timer16_0_counter[2];
extern volatile uint32_t timer16_1_counter[2];

uint8_t  *ptr = (uint8_t *)0x10003000;
uint8_t eeprom_result_data_array[4096]={0};

void fail(int er);
void pass(void);

void menu(void);
void CopyInterruptToSRAM(void);
void ClearStringBuffer(void);
uint8_t get_key( void );

#define SYSTICK_DELAY		(SystemCoreClock/1000)

#define TEST_INTERRUPT 1
#define MENU_SIZE 9

uint8_t* menu_array[MENU_SIZE] = {
(uint8_t*)"\r\nLPC11U6xx IAP Test\r\n",
(uint8_t*)"'q' - 32 BIT EEPROM write\r\n",
(uint8_t*)"'z' - 32 BIT EEPROM read\r\n",
(uint8_t*)"'r' - Read Part ID\r\n",
(uint8_t*)"'b' - Read Boot Code Version\r\n",
(uint8_t*)"'s' - Erase sector\r\n",
(uint8_t*)"'w' - Copy RAM to flash \r\n",
(uint8_t*)"'c' - Compare \r\n",
(uint8_t*)"'e' - Erase page\r\n",
};
typedef enum {
	EEPROM_32_BIT_WRITE = 'q',
	EEPROM_32_BIT_READ = 'z',
	CMD_READ_ID = 'r',
	CMD_READ_BOOT = 'b',
	CMD_ERASE_SECTOR = 's',
	CMD_WRITE = 'w',
	CMD_COMPARE = 'c',
	CMD_ERASE_PAGE = 'e',
}commandEnum;
volatile uint32_t UARTLoopbackCount = 0;
const char ascii[] = "0123456789ABCDEF";
char lineBuff[80], valBuff[80];
uint32_t *bootFirstByte, *bootSecondByte, bootTemp, bootTemp1;
static char cmd;
uint8_t rxCallbackBuf[8];

/*********************************************************************//**
 * @brief        Copy the interrupt vector on the Flash to SRAM
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void CopyInterruptToSRAM(void)
{
	unsigned int * flashPtr, * ramPtr;
  unsigned int * uLimit = (unsigned int *) 0x200;

		ramPtr = (unsigned int *)0x10000000;	//load RAM starting at 0x10000000,
		flashPtr = (unsigned int *)0x00;			//start of interrupt vector table
	  while(flashPtr < uLimit)
			{
				*ramPtr = *flashPtr;
				ramPtr++;
				flashPtr++;
			}
}

/* Main Program */
int main (void)
{
	uint32_t i,iapTemp;
	uint32_t *part_id;
	uint32_t IAP_result;
  uint32_t pa1, result[5];

  SystemCoreClockUpdate();
	#if TEST_INTERRUPT
	CopyInterruptToSRAM();		//remap interrupt vector to SRAM
	LPC_SYSCON->SYSMEMREMAP = 0x1;	//change memory map
		//Enable GPIO domain
	LPC_SYSCON->SYSAHBCLKCTRL |= (1<<6);
	//Setup PIO2_5 as GPIO to toggle LED
	LPC_IOCON->PIO2_5 &= ~(3<<3);
	LPC_GPIO_PORT->DIR[2] |= 1<<5;

	/* Called for system library in core_cmx.h(x=0 or 3). */
  SysTick_Config( SYSTICK_DELAY );
	#endif

	UARTformat(1,0,1);
	UARTInit(9600,48000000);


	menu();


	while(1)
	{

		
		ClearStringBuffer();
		cmd = get_key();
		switch(cmd)
		{

			case EEPROM_32_BIT_WRITE :
				//check 32-bit EEPROM program
				   for(i = 0; i < EEPROM_SIZE; i++){
				     *(ptr+i) = *(uint8_t *)(0x1FFF1000 + i);
				     eeprom_result_data_array[i] = ~*(ptr+i);
				   }
					 IAP_result = u32EEPROM_WRITE(0x0, (uint32_t)ptr, EEPROM_SIZE);
					 if (IAP_result == 0)
					 {
						 print_string((uint8_t*)"32 Bit EEPROM write successful \r\n");
					 }
					 else
					 {
							print_string((uint8_t*)"32 Bit EEPROM write failed \r\n");
					 }
					 break;

	case EEPROM_32_BIT_READ :
			IAP_result = u32EEPROM_READ(0x0,(uint32_t)eeprom_result_data_array,EEPROM_SIZE);
			if (IAP_result == 0)
					 {
						 print_string((uint8_t*)"32 Bit EEPROM read successful \r\n");
					 }
					 else
					 {
							print_string((uint8_t*)"32 Bit EEPROM read failed \r\n");
					 }
					 break;
  case CMD_READ_ID:
			/* Part ID */
			part_id = &iapTemp;
			IAP_result = u32IAP_ReadPartID(part_id);
			if (IAP_result == 0)
			{		sprintf(lineBuff, " Part ID is %c%c%c%c%c%c%c%c\r\n", ascii[((uint8_t)(iapTemp>>24))>>4],ascii[((uint8_t)(iapTemp>>24))& 0x0F],
					ascii[((uint8_t)(iapTemp>>16))>>4], ascii[((uint8_t)(iapTemp>>16))& 0x0F],
					ascii[((uint8_t)(iapTemp>>8))>>4], ascii[((uint8_t)(iapTemp>>8))& 0x0F],
					ascii[((uint8_t)iapTemp>>4)], ascii[((uint8_t)iapTemp& 0x0F)]);
					print_string((uint8_t*)lineBuff);
			}
		else
			{
			/* Invalid part id */
					print_string((uint8_t*)"Invalid Part ID \r\n");
			}
			break;
	case CMD_READ_BOOT:
			/* Boot version */
			bootFirstByte = &bootTemp;
			bootSecondByte = &bootTemp1;
			IAP_result = u32IAP_ReadBootVersion(bootFirstByte,bootSecondByte);

			if (IAP_result != 0)
			{
				/* Invalid bootcode rev */
				print_string((uint8_t*)"Read Boot Version Failed \r\n");
			}
			else
			{
					sprintf(lineBuff, "Boot code version # is %d.%d\r\n",bootTemp,bootTemp1);
					print_string((uint8_t*)lineBuff);

			}
			break;
	case CMD_ERASE_SECTOR:
				//erase sector 25~28
				IAP_result = u32IAP_PrepareSectors(25, 28);
				if (IAP_result != 0)
				{
					print_string((uint8_t*)"Erase Failed\r\n");
				}
				else
				{
				IAP_result = u32IAP_EraseSectors(25, 28);
				if (IAP_result != 0)
				{
					print_string((uint8_t*)"Erase Failed\r\n");
				}
				else
				{
					print_string((uint8_t*)"Erase Completed\r\n");
					// Blank check sectors
					IAP_result = u32IAP_BlankCheckSectors(25, 28, result);
					if (IAP_result != 0)
					{
						print_string((uint8_t*)"Blank check failed\r\n");
					}
					else
					{
						print_string((uint8_t*)"Blank check completed\r\n");
					}
				}
			}
			break;

		case CMD_WRITE :
				//program 1024 bytes to address 0x012100
				for(i = 0; i < 1024; i++)
					*(ptr+i) = *(uint8_t *)(SRC1 + i);
				IAP_result = u32IAP_PrepareSectors(18, 18);
				if (IAP_result != 0)
				{
					print_string((uint8_t*)"Write failed\r\n");
				}
				else
				{
					IAP_result = u32IAP_CopyRAMToFlash(DST1, (uint32_t)ptr, 1024);
					if (IAP_result != 0)
					{
						print_string((uint8_t*)"Write failed\r\n");
					}
					else
					{
						   for(i = 0; i < 1024; i++)
     *(ptr+i) = *(uint8_t *)(0x1FFF2100 + i);

						 IAP_result = u32IAP_Compare(DST1,(uint32_t)ptr, 1024, &pa1);
								if (IAP_result != 0)
								{
									print_string((uint8_t*)"Write failed.Contents not the same on comparison");
								}
								else
								{
									print_string((uint8_t*)"Write Completed successfully\r\n");
								}
					}
				}
				break;

  case CMD_COMPARE:
		for(i = 0; i < 1024; i++)
     *(ptr+i) = *(uint8_t *)(0x1FFF2100 + i);
		//check result
		IAP_result = u32IAP_Compare(0x012100, (uint32_t)ptr, 1024, &pa1);
		if (IAP_result != 0)
		{
			print_string((uint8_t*)"Contents in the memory locations are not the same");
		}
		else
		{
			print_string((uint8_t*)"Contents in the memory locations are the same");
		}
		break;
	
	case CMD_ERASE_PAGE:
		//page erase
		//page 289
		IAP_result = u32IAP_PrepareSectors(18, 18);
		if (IAP_result != 0)
		{
			print_string((uint8_t*)" Erase failed");
		}
		else
		{
			IAP_result = u32IAP_ErasePage(289, 289);
			if (IAP_result != 0)
			{
				print_string((uint8_t*)"Erase failed");
			}
			else
			{
				print_string((uint8_t*)"Page erase successful");
			}
		}
		break;

	default:
			print_string((uint8_t*)"Invalid command"); break;
}
menu();
}
}
/*********************************************************************//**
 * @brief        Clear the char buffer to output to UART
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void ClearStringBuffer(void)
{
	int i;
	for (i =0; i<80; i++)
	{
		valBuff[i] = 0x00;
		lineBuff[i] = 0x00;
	}
}

/*********************************************************************//**
 * @brief        Display the menu through UART
 *
 * @param[in]    void
 *
 * @return       void
 *
 **********************************************************************/
void menu()
{
	uint32_t i;

	for (i=0; i<MENU_SIZE; i++)
	{
		UARTSend(menu_array[i]);
	}
}
